<?php
/**
 * Biller_data table controller
 *
 * Used to perform operations on biller_data table
 *
 * @author Mario Vallejo <wappsi.desarrollomovil01@gmail.com>
*/
namespace App\Controllers;

use App\Models\SmaBillerDataModel;
use CodeIgniter\RESTful\ResourceController;

class SmaBillerData extends ResourceController
{

    /**
     * Get changes on biller_data table
     * 
     * Returns sma_biller_data changes for a given biller_id.
     *
     * @return HttpResponse
     */
    public function sync()
    {

        $rules = [
            'biller_id' => 'required',
        ];
        $this->request;
        $message = [

            'biller_id' => [
                'required' => 'Debe suministrar el biller_id a usar para filtrar los registros',
            ],
        ];
        // echo $this->request->();

        if (!$this->validate($rules, $message)) {

            // if validation fails

            $response = [
                'status' => 400,
                'error' => true,
                'message' => $this->validator->getErrors(),
                'data' => [],
                'date' => $this->request->getVar('last_sync'),
            ];

            return $this->respondCreated($response);

        } else {
            $this->setModel(new SmaBillerDataModel($this->dbGroupFromRequest()));

            // validate
            $list = [];

            if ($this->request->getVar('biller_id') != 0) {

                $list = $this->getBillerAllData($this->request->getVar('biller_id'));

                $list = $this->renameKey($list, 'id', 'id_cloud');

                $response = [
                    'status' => 200,
                    'error' => false,
                    'message' => 'Exito',
                    'server_date_time' => $this->getServerDateTime(),
                    'data' => $list,
                ];

                return $this->respondCreated($response);
            } else {
                $response = [
                    'status' => 400,
                    'error' => true,
                    'message' => 'biller_id invalido',
                    'server_date_time' => '',
                    'data' => [],
                ];

                return $this->respondCreated($response);
            }

        }

    }

    /**
     * Get biller_data data 
     * 
     * Returns array list selected rows by id of a table
     * 
     * 
     * @param int $id biller_data id
     * 
     * @return array If array is empty, not data found for  given biller id
     */
    public function getBillerAllData($id)
    {

        // $mysqltime = date ('Y-m-d H:i:s', $this->request->getVar('last_sync'));
        $rows_list = $this->model->getSelectedRows(['biller_id' => $id]);

        return $rows_list;
    }

    /**
     * Returns selected data in sma_companies, $column field is a string for select
     * who determines wich column return
     *
     * @param int $id   biller_data id
     * @param string $column    biller_data column name of interest
     * @return mixed
     */
    public static function getBillerData($id, $column,String $dbGroup)
    {
        $model = new SmaBillerDataModel($dbGroup);
        $res = $model->getColumnsData('biller_id', $id, $column);
        return $res->{$column} ?? null;
    }

    /**
     * Get default biller customer
     * 
     *  Given a biller_id, returns it customer default value
     * 
     * @param int  $biller_id   Biller id 
     *
     * @return int If returns 1, maybe it's because default biller weren't found.
     */
    public static function getBillerDefaultCustomer(int $biller_id,String $dbGroup)
    {
        try {
            $model = new SmaBillerDataModel($dbGroup);

            $customerId = $model->select('default_customer_id')
                ->where(['biller_id' => $biller_id]);
            return $customerId['default_customer_id'];
        } catch (\Throwable $th) {
            return 1;
        }
    }

    /**
     * Minimun order sale value allowed for biller
     *
     * Returns value stored on column min_sale_ammount on biller_data
     * 
     * 
     * @param int  $biller_id   Biller id
     *
     *
     * @return  int If returns 0, maybe value isn't define
     * @author Mario Vallejo <wappsi.desarrollomovil01@gmail.com>
    */
    static public function billerMinOSaleAmmount(int $biller_id, String $dbgroup){
        $model = new SmaBillerDataModel($dbgroup);
        $billerData =  $model->select("min_sale_amount")
                            ->where("biller_id",$biller_id)
                            ->first();
        return intval($billerData["min_sale_amount"]??0);
    }
    

}
